/*==============================================================
  Filename: cvsGraphCxt3D2v03.js
  Rev: 3
  By: A.R.Collins
  Description: A basic 3D graphics interface for the canvas
  element.
  License: Released into the public domain latest version at
  <http://www/arc.id.au/>
  Requires:
  - IE canvas emulator 'excanvas-modified.js' from
  <http://www.extjs.com/playpen/tm/excanvas-patch/>.
  - Text support 'canvastext.js' from
  <http://www.federated.com/~jim/canvastext/>
  - color parser 'rgbaColor.js'
  - cvs3DLib.js

  Date   |Description                                      |By
  --------------------------------------------------------------
  26Aug10 Rev 1.00 First release based on cvs3DGraphCtx2v05 ARC
  04Sep10 Include 3D state objects for use with timeline.js
          for animation                                     ARC
  11Sep10 bugfix: fixed how arguments become an array       ARC
  12Sep10 Separated movement and rendering of Shape3D
          Updated for renamed functions in cvs3DLib-03      ARC
          Make resetTransform a cvsGraphCtx3d method        ARC
  14Sep10 bugfix: ctx not updated in dup                    ARC
  09Oct10 bugfix: closepath when rendering any moveTo       ARC
  10Oct10 Rev 2.00 Added 3D text support                    ARC
  15Oct10 bugfix: yOfs signnot flip in svgPath3D            ARC
  15Oct10 Move svg & arc support functions here from svgLib
          bugfix: regexp didn't handle exponential notation
          Force svgPath to start with 'M'                   ARC
  06Dec10 bugfix: clearCanvas not using rawHeight
                  dupCtx ommitted some properties           ARC
  ==============================================================*/

  var _resized = new Array();   // keep track of which canvases are initialised
  var _busy = new Array();      // index by canvas id each element is a busy flag

  function CvsGraphCtx3D(canvasId)
  {
    this.cId = canvasId;
    this.cnvs = document.getElementById(this.cId);
    this.rawWidth = this.cnvs.offsetWidth;
    this.rawHeight = this.cnvs.offsetHeight;
    this.aRatio = this.rawWidth/this.rawHeight;

    if (!(this.cId in _resized))
    {
    /* make canvas native aspect ratio equal style box aspect ratio.
       only do this once for each canvas as it clears the canvas.
       A second graph to be drawn will erase the firt graph if the canvas
       width and height attributes are reset */
      /* Note: rawWidth and rawHeight are floats, assignment to ints will truncate */
        this.cnvs.setAttribute('width', this.rawWidth);     // this actually reset the number of graphics pixels
        this.cnvs.setAttribute('height', this.rawHeight);   // use this instead of style to match emulator

      /* create an element in associative array for this canvas
         element's existance is the test that resize has been done.
         Could have used the existance of the busy flag but lets use
         separate code for separate function */
      _resized[this.cId]= true;
    }

    if (!(this.cId in _busy))
    {
      /* this code is only executed for the first graph on each canvas
         create a busy flag for the graphics engine to prvent asynchronous
         drawing when ctx may be distorted during another call. All asynchronous
         drawing functions must be re-callable with builtin timeout and only execute
         when ctx !busy.
       */
      _busy[this.cId] = false;
    }

    this.ctx = this.cnvs.getContext('2d');
    this.ctx.save();

    this.vpW = this.rawWidth;         // vp width in pixels (default to full canvas size)
    this.vpH = this.rawHeight;        // vp height in pixels
    this.vpLLx = 0;                   // vp lower left from canvas left in pixels
    this.vpLLy = this.rawHeight;      // vp lower left from canvas top
    this.xscl = this.rawWidth/100;    // world x axis scale factor, default canvas width = 100 native units
    this.yscl = -this.rawWidth/100;   // world y axis scale factor, default +ve up and canavs height =100*aspect ratio (square pixels)
    this.xoffset = 0;                 // world x origin offset from viewport left in pixels
    this.yoffset = 0;                 // world y origin offset from viewport bottom in pixels
                                      // *** to move to world coord x ***
                                      // 1. from pixel x origin (canvas left) add vpLLx (gets to viewport left)
                                      // 2. add xoffset to get to pixel location of world x origin
                                      // 3. add x*xscl pixels to get to world x location.
                                      // ==> x (in world coords) == vpLLx + xoffset + x*xscl (pixels location of canvas)
                                      // ==> y (in world coords) == vpLLy + yoffset + y*xscl (pixels location of canvas)

    this.penCol = "rgba(0, 0, 0, 1.0)";        // black
    this.penWid = 1;             // pixels
    this.bkgCol = "rgba(255, 255, 255, 1.0)";  // white
    this.fontSize = 10;          // 10pt

    this.penX = 0;   // pen position in world coordinates
    this.penY = 0;

    this.viewpointDistance = 500;
    this.lightSource = {x:0, y:100, z:0};
  }

  // copy the basic graphics context values (for an overlay)
  
CvsGraphCtx3D.prototype.dupCtx = function(src_graphCtx)
  {
    // copy all the graphics context parameters into the overlay ctx.
    this.rawWidth = src_graphCtx.rawWidth;
    this.rawHeight = src_graphCtx.rawHeight;
    this.aRatio = src_graphCtx.aRatio;
    this.vpW = src_graphCtx.vpW;          // vp width in pixels (default to full canvas size)
    this.vpH = src_graphCtx.vpH;          // vp height in pixels
    this.vpLLx = src_graphCtx.vpLLx;      // vp lower left from canvas left in pixels
    this.vpLLy = src_graphCtx.vpLLy;      // vp lower left from canvas top
    this.xscl = src_graphCtx.xscl;        // world x axis scale factor
    this.yscl = src_graphCtx.yscl;        // world y axis scale factor
    this.xoffset = src_graphCtx.xoffset;  // world x origin offset from viewport left in pixels
    this.yoffset = src_graphCtx.yoffset;  // world y origin offset from viewport bottom in pixels
    this.penCol = src_graphCtx.penCol.slice(0);
    this.penWid = src_graphCtx.penWid;    // pixels
    this.bkgCol = src_graphCtx.bkgCol.slice(0);
    this.fontSize = src_graphCtx.fontSize;
    this.penX = src_graphCtx.penX;
    this.penY = src_graphCtx.penY;
    this.viewpointDistance = src_graphCtx.viewpointDistance;
    this.lightSource.x = src_graphCtx.lightSource.x;
    this.lightSource.y = src_graphCtx.lightSource.y;
    this.lightSource.z = src_graphCtx.lightSource.z;

    this._setCtx();
  }

 
 CvsGraphCtx3D.prototype._setCtx = function()
  {
    // often used in the library calls as the penCol etc may have been set by assignment rather than setPenColor()
    this.ctx.fillStyle = this.penCol;
    this.ctx.lineWidth = this.penWid;
    this.ctx.strokeStyle = this.penCol;
  }

  
CvsGraphCtx3D.prototype.clearCanvas = function(fillColor)
  {
    /* clearCanvas() sets the background color and fills the canvas to that color */
    var color;

    if (fillColor != undefined)
    {
      color = new RGBAColor(fillColor);

      if (color.ok)
      {
        this.bkgCol = color.toRGBA();
        this.ctx.fillStyle = this.bkgCol;
        this.ctx.fillRect(0, 0, this.rawWidth, this.rawHeight);
      }
      else
      {
        this.ctx.clearRect(0, 0, this.rawWidth, this.rawHeight);
        // all drawing erased
        // but all global graphics contexts remain intact
      }
    }
    else
    {
      this.ctx.clearRect(0, 0, this.rawWidth, this.rawHeight);
      // all drawing erased
      // but all global graphics contexts remain intact
    }
  }

 
CvsGraphCtx3D.prototype.setWorldCoords3D = function(leftX, lowerY, spanX)
  {
    if (spanX >0)
    {
      this.xscl = this.vpW/spanX;
      this.yscl = -this.xscl;
      this.xoffset = -leftX*this.xscl;
      this.yoffset = -lowerY*this.yscl;
    }
    else
    {
      this.xscl = this.rawWidth/100;    // makes xaxis = 100 native units
      this.yscl = -this.rawWidth/100;   // makes yaxis = 100*aspect ratio ie. square pixels
      this.xoffset = 0;
      this.yoffset = 0;
    }
    // world coords have changed, reset pen world coords
		this.penX = 0;
		this.penY = 0;
  }

  
CvsGraphCtx3D.prototype.setPenColor = function(color)
  {
    var newCol = new RGBAColor(color);

    if (newCol.ok)
      this.penCol = newCol.toRGBA();    // if no color passed then just restore this graph's pen color

    this.ctx.strokeStyle = this.penCol;
    this.ctx.fillStyle = this.penCol;
  }

  
CvsGraphCtx3D.prototype.setPenWidth = function(w)    // w in screen px
  {
    if (typeof w != "undefined")
      this.penWid = w;

    this.ctx.lineWidth = this.penWid;
  }

  
CvsGraphCtx3D.prototype.setViewpointDistance = function(d)    // d in world coords
  {
    if (d > 0)
      this.viewpointDistance = d;
  }

  
CvsGraphCtx3D.prototype.setLightSource = function(x, y, z)    // x, y, z in world coords
  {
    if ((x != undefined)&&(y != undefined)&&(z != undefined))
    {
      this.lightSource.x = x;
      this.lightSource.y = y;
      this.lightSource.z = z;
    }
  }

  
CvsGraphCtx3D.prototype.polyLine3D = function(data, color) // data is data[x0,y0, x1,y1 ...]
  {
    var cmdObj;
    var commands = [];
    var cPts = [];
    var ep = new Point(data[0], data[1], 0);
    var obj3D;

    cmdObj = new DrawCmd3D('moveTo', cPts, ep);
    commands.push(cmdObj);
    data.splice(0, 2);      // delete the 2 data from the front of the array
    while (data.length>0)
    {
      cPts = [];
      ep = new Point(data[0], data[1], 0);
      cmdObj = new DrawCmd3D('lineTo', cPts, ep); // any coord pair after first move is regarded as line
      commands.push(cmdObj);
      data.splice(0, 2);
    }

    var newCol = new RGBAColor(color);
    if (newCol.ok)
    {
      obj3D = new Path3D(commands, false, newCol.toRGBA());     // closed = false
    }
    else
      obj3D = new Path3D(commands, false);     // pen Color will be used

    return obj3D;
  }

  
CvsGraphCtx3D.prototype.polygon3D = function(data, fillColor) // data is data[n][2]
  {
    /* Major difference from drawing a polyLine is
     * that the extra line segment is drawn from last data point back to the first
     * to ensure a closed shape that can be filled.
     * If fillColor is specified the polygon is filled with this color
     * If fillColor is missing the polygon is draw as an outline in the current pen color
     */
    var cmdObj;
    var commands = [];
    var cPts = [];
    var ep = new Point(data[0], data[1], 0);
    var obj3D;

    cmdObj = new DrawCmd3D('moveTo', cPts, ep);
    commands.push(cmdObj);
    data.splice(0, 2);      // delete the 2 data from the front of the array
    while (data.length>0)
    {
      cPts = [];
      ep = new Point(data[0], data[1], 0);
      cmdObj = new DrawCmd3D('lineTo', cPts, ep); // any coord pair after first move is regarded as line
      commands.push(cmdObj);
      data.splice(0, 2);
    }
    cmdObj = new DrawCmd3D('closePath');
    commands.push(cmdObj);

    var newCol = new RGBAColor(fillColor);
    if (newCol.ok)
    {
      obj3D = new Path3D(commands, true, newCol.toRGBA());     // closed = true
    }
    else
    {
      // just stroke the polygon
      obj3D = new Path3D(commands, false);     // closed = false
    }

    return obj3D;
  }

  
CvsGraphCtx3D.prototype.svgPath3D = function(svgData, x, y, scale, fillColor)
  {
    var commands = this.svgPathTo3D(svgData, x, -y, scale);    // set of canvas cmds as strings with array of parameters
    var obj3D;

    var newCol = new RGBAColor(fillColor);
    if (newCol.ok)
    {
      obj3D = new Path3D(commands, true, newCol.toRGBA());     // closed = true
    }
    else
    {
      // just stroke the path
      obj3D = new Path3D(commands, false);     // closed = false
    }

    return obj3D;
  }

  
CvsGraphCtx3D.prototype.rect3D = function(x, y, w, h, fillColor)
  {
    var cmdObj;
    var commands = [];
    var cPts = [];
    var ep = new Point(x, y, 0);
    var obj3D;

    // start the path at x,y traverse CW the normal will be in -z direction (out of screen)
    cmdObj = new DrawCmd3D('moveTo', cPts, ep);
    commands.push(cmdObj);

    cPts = [];
    ep = new Point(x, y+h, 0);
    cmdObj = new DrawCmd3D('lineTo', cPts, ep); // any coord pair after first move is regarded as line
    commands.push(cmdObj);

    cPts = [];
    ep = new Point(x+w, y+h, 0);
    cmdObj = new DrawCmd3D('lineTo', cPts, ep); // any coord pair after first move is regarded as line
    commands.push(cmdObj);

    cPts = [];
    ep = new Point(x+w, y, 0);
    cmdObj = new DrawCmd3D('lineTo', cPts, ep); // any coord pair after first move is regarded as line
    commands.push(cmdObj);

    cmdObj = new DrawCmd3D('closePath');
    commands.push(cmdObj);

    var newCol = new RGBAColor(fillColor);
    if (newCol.ok)
    {
      obj3D = new Path3D(commands, true, newCol.toRGBA());     // closed = true
    }
    else
    {
      // just stroke the rectangle
      obj3D = new Path3D(commands, false);     // closed = false
    }

    return obj3D;
  }

  
CvsGraphCtx3D.prototype.arc3D = function(cx, cy, r, startAngle, endAngle, antiClockwise, fillColor)
  {
    var svgStr = genSvgArc(cx, cy, r, startAngle, endAngle, antiClockwise);
    var commands = this.svgPathTo3D(svgStr, 0, 0);    // convert svg to bezier curves i.e arc to bezier
    var obj3D;

    var newCol = new RGBAColor(fillColor);
    if (newCol.ok)
    {
      obj3D = new Path3D(commands, true, newCol.toRGBA());     // closed = true
    }
    else
    {
      // just stroke the arc using penCol
      obj3D = new Path3D(commands, false);     // closed = false
    }

    return obj3D;
  }

  
CvsGraphCtx3D.prototype.text3D = function(str, x, y, lorg, size, color)
  {
    var p3D;
    var cmdObj;
    var commands = [];
    var cPts = [];
    var ep;
    var mag = size / 25.0;    // size is worlds coords scale to stroed font size
    var xLofs = 0;
    var yLofs = 0;  /* label origin offsets */

    var strLen = CanvasTextFunctions.measure(0, size, str);

    /* Note: char cell is 33 pixels high, char size is 21 pixels (0 to 21), decenders go to -7 to 21.
       passing 'size' to text function scales char height by size/25.
       So reference height for vertically alignment is charHeight = 21/25 (=0.84) of the fontSize. */
    if (lorg != undefined)
    switch (lorg)
    {
      case 1:
      default:
        xLofs = 0;
        yLofs = 0.84*size;
        break;
      case 2:
        xLofs = 0.5*strLen;
        yLofs = 0.84*size;
        break;
      case 3:
        xLofs = strLen;
        yLofs = 0.84*size;
        break;
      case 4:
        xLofs = 0;
        yLofs = 0.42*size;
        break;
      case 5:
        xLofs = 0.5*strLen;
        yLofs = 0.42*size;
        break;
      case 6:
        xLofs = strLen;
        yLofs = 0.42*size;
        break;
      case 7:
        xLofs = 0;
        yLofs = 0;
        break;
      case 8:
        xLofs = 0.5*strLen;
        yLofs = 0;
        break;
      case 9:
        xLofs = strLen;
        yLofs = 0;
        break;
    }

    var dx = -xLofs;
    var dy = -yLofs;

    for (var i = 0; i < str.length; i++)
    {
    	var c = CanvasTextFunctions.letter(str.charAt(i));
    	if (!c)
        continue;

    	var penUp = 1;

    	for (var j = 0; j < c.points.length; j++)
      {
    	  var a = c.points[j];
    	  if ((a[0] == -1) && (a[1] == -1))
        {
      		penUp = 1;
      		continue;
    	  }
    	  if (penUp == 1)
        {
          cPts = [];
          ep = new Point(dx + a[0]*mag, dy + a[1]*mag, 0);
          cmdObj = new DrawCmd3D('moveTo', cPts, ep);
          commands.push(cmdObj);
      		penUp = 0;
    	  } else {
          cPts = [];
          ep = new Point(dx + a[0]*mag, dy + a[1]*mag, 0);
          cmdObj = new DrawCmd3D('lineTo', cPts, ep); // any coord pair after first move is regarded as line
          commands.push(cmdObj);
     	  }
    	}

    	dx += c.width*mag;
    }

    var newCol = new RGBAColor(color);
    if (newCol.ok)
    {
      p3D = new Path3D(commands, false, newCol.toRGBA());     // closed = false
    }
    else
    {
      // stroke will use the current penCol
      p3D = new Path3D(commands, false);
    }

    // now move the text object to the right place
    p3D.translate(x, y, 0);

    return p3D;
  }

  
CvsGraphCtx3D.prototype.renderPath3D = function(pg, plotNormals)
  {
    var i, j, k;

    // do the 3D projection on the Path3D vertices
    for (i=0; i<pg.ppPts.length; i++)
    {
      pg.project3D(this.viewpointDistance, pg.ppPts[i]);  // apply perspective to vertices
    }
    pg.project3D(this.viewpointDistance, pg.origin);  // apply perspective to origin
    pg.project3D(this.viewpointDistance, pg.normal);  // apply perspective to normal

    for(j=0; j<pg.drawCmds.length; j++)   // step through the draw segments
    {
      pg.drawData[j] = new DrawCmd(pg.drawCmds[j].drawFn);   // this is a 2D drawCmd
      for (k=0; k<pg.drawCmds[j].cPts.length; k++)   // extract flattened 2D coords from 3D Points
      {
        pg.drawData[j].parms[2*k] = pg.drawCmds[j].cPts[k].fx;
        pg.drawData[j].parms[2*k+1] = pg.drawCmds[j].cPts[k].fy;
      }
      // add the end point (check it exists since 'closePath' has no end point)
      if (pg.drawCmds[j].ep != undefined)
      {
        pg.drawData[j].parms[2*k] = pg.drawCmds[j].ep.fx;
        pg.drawData[j].parms[2*k+1] = pg.drawCmds[j].ep.fy;
      }
    }

    var pgcol = pg.color || this.penCol;
    this.ctx.fillStyle = pgcol;
    this.ctx.strokeStyle = pgcol;

    this.ctx.beginPath();
    // step through the Path3D drawData array of commands and draw each one
    for (j=0; j < pg.drawData.length; j++)
    {
      // convert all parms to world coords
      for (k=0; k<pg.drawData[j].parms.length; k+=2)   // step thru the coords in x,y pairs
      {
        pg.drawData[j].parms[k] = this.vpLLx+this.xoffset+pg.drawData[j].parms[k]*this.xscl;
        pg.drawData[j].parms[k+1] = this.vpLLy+this.yoffset+pg.drawData[j].parms[k+1]*this.yscl;
      }
      // to make filling of paths more predictable, fill before all 'moveTo'
      if ((j>0)&&(pg.drawData[j].drawFn == 'moveTo'))
      {
        if (pg.closed)
        {
          this.ctx.closePath();
          this.ctx.fill();
        }
        this.ctx.stroke();    // stroke outline
        this.ctx.beginPath();
      }

      this.ctx[pg.drawData[j].drawFn].apply(this.ctx, pg.drawData[j].parms);   // execute canvas draw command
    }
    if (pg.closed)
      this.ctx.fill();
    this.ctx.stroke();    // stroke outline

    if (plotNormals)     // draw the normal
    {
      var savCol = this.penCol;

      // convert the origin and normal too
      var ox = this.vpLLx+this.xoffset+pg.origin.fx*this.xscl;
      var oy = this.vpLLy+this.yoffset+pg.origin.fy*this.yscl;
      var nx = this.vpLLx+this.xoffset+pg.normal.fx*this.xscl;
      var ny = this.vpLLy+this.yoffset+pg.normal.fy*this.yscl;

      if (pg.origin.tz < pg.normal.tz)
        this.setPenColor("red");        // pointing away from viewer
      else
        this.setPenColor("green");      // pointing toward viewer


      this.ctx.beginPath();
      this.ctx.moveTo(ox, oy);
      this.ctx.lineTo(nx, ny);
      this.ctx.stroke();

      this.setPenColor(savCol);      // restore pencol
    }
  }

  
CvsGraphCtx3D.prototype.groupPaths = function()
  {
    var args = Array.prototype.slice.call(arguments); // grab array of arguments
    var grp = new Group3D();

    // steps through the arguments
    for (var i=0; i<args.length; i++)
    {
      if (args[i] instanceof Array)          // array of Path3D
      {
        for (var j=0; j<args[i].length; j++)
          grp.addPath3D(args[i][j]);
      }
      else       // single Path3D
      {
        grp.addPath3D(args[i]);
      }
    }

    return grp
  }

/* =======================================================================
 * groupClosedPaths
 * Add Path3D or array of Path3Ds to the backcullPaths
 * Paths added to a group with this method will only be rendered if their
 * normal points toward the viewpoint. This is useful only for closed
 * shapes, as these backward facing panels won't be seen. This is provided
 * just to save processing time by eliminating unneccessary drawing.
 * -----------------------------------------------------------------------
 */
  
CvsGraphCtx3D.prototype.groupClosedPaths = function()
  {
    var args = Array.prototype.slice.call(arguments); // grab array of arguments
    var grp = new Group3D();

    // steps through the arguments
    for (var i=0; i<args.length; i++)
    {
      if (args[i] instanceof Array)          // array of Path3D
      {
        for (var j=0; j<args[i].length; j++)
          grp.addBackcullPath3D(args[i][j]);
      }
      else       // single Path3D
      {
        grp.addBackcullPath3D(args[i]);
      }
    }

    return grp
  }

/* =======================================================================
 * groupByRotation
 * Given a Path3D which will define a profile, this function
 * makes a set of copies of this profile each rotated about an axis, like
 * the segments of an orange. The set of closed Path3Ds are created
 * by each segment (drawCmd) in the profile being joined to the
 * corresponding end points of the segment in the adjacent
 * copy of the profile (which has been rotated about the axis).
 * Parameters:
 * path: Path3D defineing the shapes profile
 * radius: distance from start of profile to the center of rotation
 * totalAngle: 360 degrees for full rotation or less
 * segments: number of segments into which totalAngle is divided
 * fillColor: HTML format color string
 * return: the Group3D that will draw the shape
 * -----------------------------------------------------------------------
 */
  
CvsGraphCtx3D.prototype.groupByRotation = function(path, radius, totalAngle, segments, fillColor)
  {
    /* To get fill to work, path sections must traversed in a consistant direction
     * CW or CCW, not by going halfway, adding move commands and extra segments.
     * Tranversing the path in one direction requires the svg path to be tranversd
     * backwards and forwards! Too hard.
     * So draw each area in two halves: across top then draw profile, diagonally
     * back to top, and the second one down profile and across the bottom and diagonally
     * back to the top. The two 'triangles' should fill
     * correctly as they are traversed in a consistant direction
     * and the profile is easy to traverse as its is always from top down.
     */

    var ppArray = [];
    var grp;

    var startX = 0;
    var startY = 0;
    var endX = 0;
    var endY = 0;
    var pp0, pp1;
    var topRim;
    var botRim;
    var botRimPP, topRimPP;
    var commands;

    radius = Math.abs(radius);
    var totA = totalAngle || 360;
    var segs = 6;
    if ((segments != undefined)&&(segments>0))
      segs = segments;
    var segAng = totA / segs;           // included angle of each segment
    var color = this.penCol;
    var newCol = new RGBAColor(fillColor);
    if (newCol.ok)
      color = newCol.toRGBA();

    var profile_0 = path.dup(); // make a copy
    var profile_1 = path.dup(); // two needed (not new reference)
    // calc origin coords given start coords of profile and the radius
    var orgX = profile_0.drawCmds[0].ep.x - radius;   // x coord of origin (center of rotation at top of profile)
    var orgY = profile_0.drawCmds[0].ep.y;            // y coord of origin, both in real 3D coords
    // move the profile to have origin at top center of rotation
    profile_0.translate(-orgX, -orgY, 0);  // 3D coords
    profile_1.translate(-orgX, -orgY, 0);  // 3D coords
    // now this profile must be rotated by the segment angle to form the other side
    profile_1.rotate(0, segAng, 0);   // rotate segment by segAng out of screen
    var topClosed = (profile_0.drawCmds[0].ep.x < 3);

    for (var n=0; n< segs; n++)
    {
      for (var m = 1; m < profile_0.drawCmds.length; m++)   // first drawCmd is moveTo, skip this
      {
        // construct the polyPath from movement down profile of 1 drawCmd
        // do this in 2 halves so we can traverse the profile top to bottom for each side
        startX = profile_0.drawCmds[m-1].ep.x;     // ep is the end point (others are control points)
        startY = profile_0.drawCmds[m-1].ep.y;
        endX = profile_0.drawCmds[m].ep.x;
        endY = profile_0.drawCmds[m].ep.y;

        pp0 = new Path3D();
        // make a topRim if profile doesn't start at centre
        if (startX > 3)
        {
          topRim = genSvgArc(0, 0, startX, segAng, 0, false);    // generate the SVG string for the arc of rotation
          // start with top rim (draw in xy), endpoint will be where this profile slice starts
          commands = this.svgPathTo3D(topRim, 0, 0, 1);    // set of canvas cmds as strings with array of parameters
          topRimPP = new Path3D(commands, true);   // closed = true
          // topRim is in xy plane must be rotated to be in xz plane to join profile
          topRimPP.rotate(-90, 0, 0);    // flip top out of screen
          topRimPP.translate(0, startY, 0);   // move down from y=0 to top of profile slice
          // add this the the empty pp0
          pp0.appendPath3D(topRimPP);
        }
        else
        {
          // construct a moveTo command from end point of last command
          var mc = new DrawCmd3D("moveTo", [], profile_0.drawCmds[m-1].ep.clone());
          pp0.addDrawCmd(mc);     // use this to start the new Path
        }
        // copy the profile drawCmd for this slice to the PolyPath
        pp0.addDrawCmd(profile_0.drawCmds[m].clone());

        // now traverse the 2nd half, down the profile_1 then across the bottom
        pp1 = new Path3D();  // start with a new polypath
        // construct a moveTo command from end point of last command
        var dc = new DrawCmd3D("moveTo", [], profile_1.drawCmds[m-1].ep.clone());
        pp1.addDrawCmd(dc);     // use this to start the new Path
        pp1.addDrawCmd(profile_1.drawCmds[m].clone());  // now add a copy of the drawCmd

        // make the bottom rim if it has any size
        if (endX > 3)
        {
          botRim = genSvgArc(0, 0, endX, -segAng, 0, true);  // -ve angle traverse CCW
          commands = this.svgPathTo3D(botRim, 0, 0, 1);    // set of canvas cmds as strings with array of parameters
          botRimPP = new Path3D(commands, true);
          // rim is in xy plane rotate to be in xz plane
          botRimPP.rotate(90, 0, 0);       // flip bottom up to be out of screen
          botRimPP.translate(0, endY, 0);   // move down from y=0 to bottom of profile
          // now concatinate with the rest
          pp1.appendPath3D(botRimPP, true);  // concatinate and delete the first moveTo to help define fill area
        }
        // now concatinate the 2nd half with the first
        pp0.appendPath3D(pp1);   // dont delete the first move this time this is 2nd half
        pp0.closed = true;         // it will only be filled if closed=true
        pp0.color = color;         // set the fill color
        // now add the complete polyPath to the array which makes the final shape

        ppArray.push(pp0);
      }
      //rotate the previously made PPs out of the way of next segment
      for (var k=0; k < ppArray.length; k++)
        ppArray[k].rotate(0, segAng, 0);
    }
    var botClosed = (endX < 3);
    if (topClosed && botClosed)  // closed path
      grp = this.groupClosedPaths(ppArray);
    else
      grp = this.groupPaths(ppArray);

    return grp
  }

  
CvsGraphCtx3D.prototype.resetTransformGroup3D = function(grp)
  {
    grp.xfrm = identityMatrix();
  }

  
CvsGraphCtx3D.prototype.rotateGroup3D = function(grp, xRot, yRot, zRot, cx, cy, cz)
  {
    if ((cx != undefined)&&(cy != undefined)&&(cz != undefined))
      grp.translation(-cx, -cy, -cz);    // put center of rotation at 0,0,0
    if ((xRot != undefined)&&(yRot != undefined)&&(zRot != undefined))
      grp.rotation(xRot, yRot, zRot);    // rotate
    if ((cx != undefined)&&(cy != undefined)&&(cz != undefined))
      grp.translation(cx, cy, cz);         // translate back
  }

  
CvsGraphCtx3D.prototype.translateGroup3D = function(grp, x, y, z)
  {
    if ((x != undefined)&&(y != undefined)&&(z != undefined))
      grp.translation(x, y, z);          // translate to final position
  }

  
CvsGraphCtx3D.prototype.renderGroup3D = function(grp)
  {
    var i, j, k;
    var pg, pgcol;

    grp.genDrawData(this.viewpointDistance, false);       // wireframe = false

    for (i=0; i < grp.pathDrawList.length; i++)
    {
      pg = grp.pathDrawList[i];   // point to each Path3D in list
      pgcol = pg.color || this.penCol;

      if (!pg.closed)    // draw as wireframe
      {
        this.ctx.strokeStyle = pgcol;
      }
      else   // solid panels
      {
        // do shading
        if (true)             // shade surfaces = true
          pgcol = pg.shadePath3D(this.viewpointDistance, this.lightSource, pgcol);
        this.ctx.fillStyle = pgcol;
        this.ctx.strokeStyle = pgcol;
      }

      this.ctx.beginPath();
      // step through the Path3D drawData array of commands and draw each one
      for (j=0; j < pg.drawData.length; j++)
      {
        // convert all parms to world coords
        for (k=0; k<pg.drawData[j].parms.length; k+=2)   // step thru the coords in x,y pairs
        {
          pg.drawData[j].parms[k] = this.vpLLx+this.xoffset+pg.drawData[j].parms[k]*this.xscl;
          pg.drawData[j].parms[k+1] = this.vpLLy+this.yoffset+pg.drawData[j].parms[k+1]*this.yscl;
        }
        // to make filling of paths more predictable, fill before all 'moveTo'
        if ((j>0)&&(pg.drawData[j].drawFn == 'moveTo'))
        {
          if (pg.closed)
          {
            this.ctx.closePath();
            this.ctx.fill();
          }
          this.ctx.stroke();    // stroke outline
          this.ctx.beginPath();
        }

        this.ctx[pg.drawData[j].drawFn].apply(this.ctx, pg.drawData[j].parms);   // render to canvas
      }
      if (pg.closed)
        this.ctx.fill();
      this.ctx.stroke();    // stroke outline

      /*
            if (true)     // draw the normals
            {
              var savCol = this.penCol;

              // convert the origin and normal too
              var ox = this.vpLLx+this.xoffset+pg.origin.fx*this.xscl;
              var oy = this.vpLLy+this.yoffset+pg.origin.fy*this.yscl;
              var nx = this.vpLLx+this.xoffset+pg.normal.fx*this.xscl;
              var ny = this.vpLLy+this.yoffset+pg.normal.fy*this.yscl;

              if (pg.origin.tz < pg.normal.tz)
                this.setPenColor("red");        // pointing away from viewer
              else
                this.setPenColor("green");      // pointing toward viewer


              this.ctx.beginPath();
              this.ctx.moveTo(ox, oy);
              this.ctx.lineTo(nx, ny);
              this.ctx.stroke();

              this.setPenColor(savCol);      // restore pencol
            }
      */

    }
  }

  
CvsGraphCtx3D.prototype.wireframeGroup3D = function(grp)
  {
    var i, j, k;

    grp.genDrawData(this.viewpointDistance, true);      // wireframe = true so no back-culling

    for (i=0; i < grp.pathDrawList.length; i++)
    {
      var pg = grp.pathDrawList[i];   // point to each Path3D in list

      if (pg.color !== null)
        this.ctx.strokeStyle = pg.color;
      else
        this.ctx.strokeStyle = this.penCol;

      this.ctx.beginPath();
      // step through the Path3D drawData array of commands and draw each one
      for (j=0; j < pg.drawData.length; j++)
      {
        // convert all parms to world coords
        for (k=0; k<pg.drawData[j].parms.length; k+=2)   // step thru the coords in x,y pairs
        {
          pg.drawData[j].parms[k] = this.vpLLx+this.xoffset+pg.drawData[j].parms[k]*this.xscl;
          pg.drawData[j].parms[k+1] = this.vpLLy+this.yoffset+pg.drawData[j].parms[k+1]*this.yscl;
        }
        this.ctx[pg.drawData[j].drawFn].apply(this.ctx, pg.drawData[j].parms);   // draw the Path3D
      }
      this.ctx.stroke();    // stroke outline
    }
  }

/* Upgraded version of the 'cake.js' compileSVGPath  */

CvsGraphCtx3D.prototype.svgPathTo3D = function(svgPath, xOfs, yOfs, scl)
{
  var xScale = scl || 1;
  var yScale = -xScale;
  var segs = svgPath.split(/(?=[a-df-z])/i);
  var x = 0;
  var y = 0;
  var cPts;        // array of control points
  var ep;          // end point
  var px,py;
  var pc;
  var cmdObj;
  var commands = [];
  for (var i=0; i<segs.length; i++)
  {
    var seg = segs[i];
    var cmdLetters = seg.match(/[a-z]/i);
    if (!cmdLetters)
      return [];
    cmd = cmdLetters[0];
      if ((i==0)&&(cmd != 'M'))   // check that the first move is absolute
        cmd = 'M';
    var coords = seg.match(/[-+]?[0-9]*\.?[0-9]+([eE][-+]?[0-9]+)?/gi);
    if (coords)
      coords = coords.map(parseFloat);
    switch(cmd)
    {
      case 'M':
        x = xScale*(xOfs+coords[0]);
        y = yScale*(yOfs+coords[1]);
        px = py = null;
        cPts = [];
        ep = new Point(x, y, 0);
        cmdObj = new DrawCmd3D('moveTo', cPts, ep);
        commands.push(cmdObj);
        coords.splice(0, 2);      // delete the 2 coords from the front of the array
        while (coords.length>0)
        {
          x = xScale*(xOfs+coords[0]);                // eqiv to muliple 'L' calls
          y = yScale*(yOfs+coords[1]);
          cPts = [];
          ep = new Point(x, y, 0);
          cmdObj = new DrawCmd3D('lineTo', cPts, ep); // any coord pair after first move is regarded as line
          commands.push(cmdObj);
          coords.splice(0, 2);
        }
        break
      case 'm':
        x += xScale*coords[0];
        y += yScale*coords[1];
        px = py = null;
        cPts = [];
        ep = new Point(x, y, 0);
        cmdObj = new DrawCmd3D('moveTo', cPts, ep);
        commands.push(cmdObj);
        coords.splice(0, 2);      // delete the 2 coords from the front of the array
        while (coords.length>0)
        {
          x += xScale*coords[0];                     // eqiv to muliple 'l' calls
          y += yScale*coords[1];
          cPts = [];
          ep = new Point(x, y, 0);
          cmdObj = new DrawCmd3D('lineTo', cPts, ep); // any coord pair after first move is regarded as line
          commands.push(cmdObj);
          coords.splice(0, 2);
        }
        break

      case 'L':
        while (coords.length>0)
        {
          x = xScale*(xOfs+coords[0]);
          y = yScale*(yOfs+coords[1]);
          cPts = [];
          ep = new Point(x, y, 0);
          cmdObj = new DrawCmd3D('lineTo', cPts, ep);
          commands.push(cmdObj);
          coords.splice(0, 2);
        }
        px = py = null;
        break
      case 'l':
        while (coords.length>0)
        {
          x += xScale*coords[0];
          y += yScale*coords[1];
          cPts = [];
          ep = new Point(x, y, 0);
          cmdObj = new DrawCmd3D('lineTo', cPts, ep);
          commands.push(cmdObj);
          coords.splice(0, 2);
        }
        px = py = null
        break
      case 'H':
        x = xScale*(xOfs+coords[0]);
        px = py = null ;
        cPts = [];
        ep = new Point(x, y, 0);
        cmdObj = new DrawCmd3D('lineTo', cPts, ep);
        commands.push(cmdObj);
        break
      case 'h':
        x += xScale*coords[0];
        px = py = null ;
        cPts = [];
        ep = new Point(x, y, 0);
        cmdObj = new DrawCmd3D('lineTo', cPts, ep);
        commands.push(cmdObj);
        break
      case 'V':
        y = yScale*(yOfs+coords[0]);
        px = py = null;
        cPts = [];
        ep = new Point(x, y, 0);
        cmdObj = new DrawCmd3D('lineTo', cPts, ep);
        commands.push(cmdObj);
        break
      case 'v':
        y += yScale*coords[0];
        px = py = null;
        cPts = [];
        ep = new Point(x, y, 0);
        cmdObj = new DrawCmd3D('lineTo', cPts, ep);
        commands.push(cmdObj);
        break
      case 'C':
        while (coords.length>0)
        {
          c1x = xScale*(xOfs+coords[0]);
          c1y = yScale*(yOfs+coords[1]);
          px = xScale*(xOfs+coords[2]);
          py = yScale*(yOfs+coords[3]);
          x = xScale*(xOfs+coords[4]);
          y = yScale*(yOfs+coords[5]);
          cPts = [];
          cPts[0] = new Point(c1x, c1y, 0);
          cPts[1] = new Point(px, py, 0);
          ep = new Point(x, y, 0);
          cmdObj = new DrawCmd3D('bezierCurveTo', cPts, ep);
          commands.push(cmdObj);
          coords.splice(0, 6);
        }
        break
      case 'c':
        while (coords.length>0)
        {
          c1x = x + xScale*coords[0];
          c1y = y + yScale*coords[1];
          px = x + xScale*coords[2];
          py = y + yScale*coords[3];
          x += xScale*coords[4];
          y += yScale*coords[5];
          cPts = [];
          cPts[0] = new Point(c1x, c1y, 0);
          cPts[1] = new Point(px, py, 0);
          ep = new Point(x, y, 0);
          cmdObj = new DrawCmd3D('bezierCurveTo', cPts, ep);
          commands.push(cmdObj);
          coords.splice(0, 6);
        }
        break
      case 'S':
        if (px == null || !pc.match(/[sc]/i))
        {
          px = x;                // already absolute coords
          py = y;
        }
        cPts = [];
        cPts[0] = new Point(x-(px-x), y-(py-y), 0);
        cPts[1] = new Point(xScale*(xOfs+coords[0]), yScale*(yOfs+coords[1]), 0);
        ep = new Point(xScale*(xOfs+coords[2]), yScale*(yOfs+coords[3]), 0);
        cmdObj = new DrawCmd3D('bezierCurveTo', cPts, ep);
        commands.push(cmdObj);
        px = xScale*(xOfs+coords[0]);
        py = yScale*(yOfs+coords[1]);
        x = xScale*(xOfs+coords[2]);
        y = yScale*(yOfs+coords[3]);
        break
      case 's':
        if (px == null || !pc.match(/[sc]/i))
        {
          px = x;
          py = y;
        }
        cPts = [];
        cPts[0] = new Point(x-(px-x), y-(py-y), 0);
        cPts[1] = new Point(x + xScale*coords[0], y + yScale*coords[1], 0);
        ep = new Point(x + xScale*coords[2], y + yScale*coords[3], 0);
        cmdObj = new DrawCmd3D('bezierCurveTo', cPts, ep);
        commands.push(cmdObj);
        px = x + xScale*coords[0];
        py = y + yScale*coords[1];
        x += xScale*coords[2];
        y += yScale*coords[3];
        break
      case 'Q':
        px = xScale*(xOfs+coords[0]);
        py = yScale*(yOfs+coords[1]);
        x = xScale*(xOfs+coords[2]);
        y = yScale*(yOfs+coords[3]);
        cPts = [];
        cPts[0] = new Point(px, py, 0);
        ep = new Point(x, y, 0);
        cmdObj = new DrawCmd3D('quadraticCurveTo', cPts, ep);
        commands.push(cmdObj);
        break
      case 'q':
        cPts = [];
        cPts[0] = new Point(x + xScale*coords[0], y + yScale*coords[1], 0);
        ep = new Point(x + xScale*coords[2], y + yScale*coords[3], 0);
        cmdObj = new DrawCmd3D('quadraticCurveTo', cPts, ep);
        commands.push(cmdObj);
        px = x + xScale*coords[0];
        py = y + yScale*coords[1];
        x += xScale*coords[2];
        y += yScale*coords[3];
        break
      case 'T':
        if (px == null || !pc.match(/[qt]/i))
        {
          px = x;
          py = y;
        }
        else
        {
          px = x-(px-x);
          py = y-(py-y);
        }
        cPts = [];
        cPts[0] = new Point(px, py, 0);
        ep = new Point(xScale*(xOfs+coords[0]), yScale*(yOfs+coords[1]), 0);
        cmdObj = new DrawCmd3D('quadraticCurveTo', cPts, ep);
        commands.push(cmdObj);
        px = x-(px-x);
        py = y-(py-y);
        x = xScale*(xOfs+coords[0]);
        y = yScale*(yOfs+coords[1]);
        break
      case 't':
        if (px == null || !pc.match(/[qt]/i))
        {
          px = x;
          py = y;
        }
        else
        {
          px = x-(px-x);
          py = y-(py-y);
        }
        cPts = [];
        cPts[0] = new Point(px, py, 0);
        ep = new Point(x + xScale*coords[0], y + yScale*coords[1], 0);
        cmdObj = new DrawCmd3D('quadraticCurveTo', cPts, ep);
        commands.push(cmdObj);
        x += xScale*coords[0];
        y += yScale*coords[1];
        break
      case 'A':
        coords[0] *= xScale;
        coords[1] *= xScale;
        coords[2] *= -1;          // rotationX: swap for CCW +ve
        // coords[3] large arc    should be ok
        coords[4] = 1 - coords[4];      // swap for CCW +ve
        coords[5] *= xScale;
        coords[5] += xOfs;
        coords[6] *= yScale;
        coords[6] += yOfs;
        var arc_segs = this.arcToBezier(x, y, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5], coords[6]);
        for (var l=0; l<arc_segs.length; l++)
        {
          cPts = [];
          cPts[0] = new Point(arc_segs[l][0], arc_segs[l][1], 0);
          cPts[1] = new Point(arc_segs[l][2], arc_segs[l][3], 0);
          ep = new Point(arc_segs[l][4], arc_segs[l][5], 0);
          cmdObj = new DrawCmd3D('bezierCurveTo', cPts, ep);
          commands.push(cmdObj);
        }
        x = coords[5];
        y = coords[6];
        break
      case 'a':
        coords[0] *= xScale;
        coords[1] *= xScale;
        coords[2] *= -1;        // rotationX: swap for CCW +ve
        // coords[3] large arc    should be ok
        coords[4] = 1 - coords[4];   // sweep: swap for CCW +ve
        coords[5] *= xScale;
        coords[5] += x;
        coords[6] *= yScale;
        coords[6] += y;
        var arc_segs = this.arcToBezier(x, y, coords[0], coords[1], coords[2], coords[3], coords[4], coords[5], coords[6]);
        for (var l=0; l<arc_segs.length; l++)
        {
          cPts = [];
          cPts[0] = new Point(arc_segs[l][0], arc_segs[l][1], 0);
          cPts[1] = new Point(arc_segs[l][2], arc_segs[l][3], 0);
          ep = new Point(arc_segs[l][4], arc_segs[l][5], 0);
          cmdObj = new DrawCmd3D('bezierCurveTo', cPts, ep);
          commands.push(cmdObj);
        }
        x = coords[5];
        y = coords[6];
        break
      case 'Z':
        cmdObj = new DrawCmd3D('closePath');
        commands.push(cmdObj);
        break
      case 'z':
        cmdObj = new DrawCmd3D('closePath');
        commands.push(cmdObj);
        break
    }
    pc = cmd     // save the previous command for possible reflected control points
  }
  return commands
}

CvsGraphCtx3D.prototype.arcToBezier = function(ox, oy, rx, ry, rotateX, large, sweep, x, y)
{
  var th = rotateX * (Math.PI/180)
  var sin_th = Math.sin(th)
  var cos_th = Math.cos(th)
  rx = Math.abs(rx)
  ry = Math.abs(ry)
  var px = cos_th * (ox - x) * 0.5 + sin_th * (oy - y) * 0.5
  var py = cos_th * (oy - y) * 0.5 - sin_th * (ox - x) * 0.5
  var pl = (px*px) / (rx*rx) + (py*py) / (ry*ry)
  if (pl > 1)
  {
    pl = Math.sqrt(pl)
    rx *= pl
    ry *= pl
  }

  var a00 = cos_th / rx
  var a01 = sin_th / rx
  var a10 = (-sin_th) / ry
  var a11 = (cos_th) / ry
  var x0 = a00 * ox + a01 * oy
  var y0 = a10 * ox + a11 * oy
  var x1 = a00 * x + a01 * y
  var y1 = a10 * x + a11 * y

  var d = (x1-x0) * (x1-x0) + (y1-y0) * (y1-y0)
  var sfactor_sq = 1 / d - 0.25
  if (sfactor_sq < 0)
    sfactor_sq = 0
  var sfactor = Math.sqrt(sfactor_sq)
  if (sweep == large)
    sfactor = -sfactor
  var xc = 0.5 * (x0 + x1) - sfactor * (y1-y0)
  var yc = 0.5 * (y0 + y1) + sfactor * (x1-x0)

  var th0 = Math.atan2(y0-yc, x0-xc)
  var th1 = Math.atan2(y1-yc, x1-xc)

  var th_arc = th1-th0
  if (th_arc < 0 && sweep == 1)
  {
    th_arc += 2*Math.PI
  }
  else if (th_arc > 0 && sweep == 0)
  {
    th_arc -= 2 * Math.PI
  }

  var segments = Math.ceil(Math.abs(th_arc / (Math.PI * 0.5 + 0.001)))
  var result = []
  for (var i=0; i<segments; i++)
  {
    var th2 = th0 + i * th_arc / segments
    var th3 = th0 + (i+1) * th_arc / segments
    result.push(this.segmentToBezier(xc, yc, th2, th3, rx, ry, sin_th, cos_th));
  }

  return result
}

CvsGraphCtx3D.prototype.segmentToBezier = function(cx, cy, th0, th1, rx, ry, sin_th, cos_th)
{
  var a00 = cos_th * rx
  var a01 = -sin_th * ry
  var a10 = sin_th * rx
  var a11 = cos_th * ry

  var th_half = 0.5 * (th1 - th0)
  var t = (8/3) * Math.sin(th_half * 0.5) * Math.sin(th_half * 0.5) / Math.sin(th_half)
  var x1 = cx + Math.cos(th0) - t * Math.sin(th0)
  var y1 = cy + Math.sin(th0) + t * Math.cos(th0)
  var x3 = cx + Math.cos(th1)
  var y3 = cy + Math.sin(th1)
  var x2 = x3 + t * Math.sin(th1)
  var y2 = y3 - t * Math.cos(th1)
  return [
            a00 * x1 + a01 * y1, a10 * x1 + a11 * y1,
            a00 * x2 + a01 * y2, a10 * x2 + a11 * y2,
            a00 * x3 + a01 * y3, a10 * x3 + a11 * y3
          ]
}

if (!Array.prototype.map)
{
  Array.prototype.map = function(fun)
  {
    var len = this.length;
    if (typeof fun != "function")
      throw new TypeError();

    var res = new Array(len);
    var thisp = arguments[1];
    for (var i = 0; i < len; i++)
    {
      if (i in this)
        res[i] = fun.call(thisp, this[i], i, this);
    }

    return res;
  };
}

/* ========================================================
 * Generate the SVG format string for a circular arc:
 * The arc center is at cx, cy. Arc starts from startAngle
 * and ends at endAngle. startAngle and endAngle are in
 * degrees. The arc radius is r (in world coords). If
 * antiClockwise is true the arc is traversed ccw, if false
 * it is traversed cw.
 *---------------------------------------------------------
 * NOTE: all this is in SVG coords y +ve down
 */
function genSvgArc(cx, cy, r, startAngle, endAngle, antiClockwise)
{
  var stRad = startAngle * Math.PI/180;
  var edRad = endAngle * Math.PI/180;

  var oy = -cy - r*Math.sin(stRad);   // coords of start point for circlular arc (y flipped) relative to center (cx,cy)
  var ox = cx + r*Math.cos(stRad);
  var ey = -cy - r*Math.sin(edRad);   // coords of start point for circlular arc (y flipped) relative to center (cx,cy)
  var ex = cx + r*Math.cos(edRad);
  var rotX = 0;     // tilt of x axis, always 0 for circular arc
  var lrgArc = 0;   // always use small so 0
  var ccw = (antiClockwise? 0 : 1);         // 0=ccw 1=cw   (flipped for this ccw +ve world)
  var delta = 0;
  var svgData;

  if (ccw)
    delta = stRad - edRad;
  else
    delta = edRad - stRad;
  if (delta < 0)
    delta += 2*Math.PI;
  if (delta > 2* Math.PI)
    delta -= 2*Math.PI;
  lrgArc = delta > Math.PI? 1: 0;

  // dont try to draw full circle or no circle (arcToBezier will crash)
  if ((Math.abs(delta) < 0.01) || (Math.abs(delta) > 2*Math.PI-0.01))
    svgData = genSvgCircle(cx, -cy, r);
  else
    svgData = "M"+ox.toFixed(3)+","+oy.toFixed(3)+"A"+r+","+r+","+0+","+lrgArc+","+ccw+","+ex.toFixed(3)+","+ey.toFixed(3);

  return svgData
}

/* ========================================================
 * Generate the svg format string for to draw a circle
 * using 4 quadBezier curves.
 * x,y is centre, radius r
 *---------------------------------------------------------
 */
function genSvgCircle(x, y, r)
{
  var m = 0.55228475;
  var c1x, c1y, c2x, c2y, ex, ey;

  ex = x;
  ey = y-r;
  var s="M"+ex+","+ey;

  // 1st quadrant
  c1x = x+m*r;
  c1y = y-r;
  c2x = x+r;
  c2y = y-m*r;
  ex = x+r;
  ey = y;
  s += "C"+c1x+","+c1y+","+c2x+","+c2y+","+ex+","+ey;
  // 2nd quadrant
  c1x = x+r;
  c1y = y+m*r;
  c2x = x+m*r;
  c2y = y+r;
  ex = x;
  ey = y+r;
  s += "C"+c1x+","+c1y+","+c2x+","+c2y+","+ex+","+ey;
  // 3rd quadrant
  c1x = x-m*r;
  c1y = y+r;
  c2x = x-r;
  c2y = y+m*r;
  ex = x-r;
  ey = y;
  s += "C"+c1x+","+c1y+","+c2x+","+c2y+","+ex+","+ey;
  // 4th quadrant
  c1x = x-r;
  c1y = y-m*r;
  c2x = x-m*r;
  c2y = y-r;
  ex = x;
  ey = y-r;
  s += "C"+c1x+","+c1y+","+c2x+","+c2y+","+ex+","+ey;

  return s
}

// ============ useful objects for animation, these are returned by 3D path functions ============
// constructor for position and rotation state for moving an object in 3D.


function State3D(x, y, z, xRot, yRot, zRot, cx, cy, cz)
{
  this.x = x;
  this.y = y;
  this.z = z;
  this.xRot = xRot;
  this.yRot = yRot;
  this.zRot = zRot;
  this.cx = cx;     // center of rotation
  this.cy = cy;
  this.cz = cz;
}

