/*======================================================================
  Filename: cvs3DLib-07.js
  Rev: 7
  By: A.R.Collins
  Description: Set of 3D drawing utilities supporting SVG format
  curves as well as straight sided shapes. The functions, objects
  and methods are designed to generate output that is in a format
  of HTML5 canvas drawing commands and their parameters.

  3D Matrix manipulation code borrows heavily from Google 3D demo.
  SVG parser is based on code from cake.js
  <http://code.google.com/p/cakejs/> many thanks.
  Modified to handle the SVG shortcuts which omit repeated
  or unambiguous command specifiers.

  Date   |Description                                               |By
  ----------------------------------------------------------------------
  31Aug10 First release, combining cvs3DUtils-13.js and
          svgPathToCanvas-08.js renamed from cvsObject3D             arc
  10Seo10 Change order of rotate to z, y, x and update comments      arc
  12Sep10 Renamed Path3D.rotateRel to rotate and translateRel to
          translate
          Remove grp.resetTransform to CvsGraphCtx3D                 arc
  13Sep10 Rewrite to remove xfrm property from paths (only ever used
          as a temporary variable. Do trasnforms in one step
          Do Group3D transforms (overwrite type) in one step to
          avoid double purpose use of  xfrm parameter.               arc
          Removed Group3D points property, use paths.ppPts instead   arc
  15Oct10 Move svg and arc stuff to cvsGraphCtx3D
          Add support for transparent colors in shading              arc
 *======================================================================*/

/* create a copy (not just a reference) of an object */
Object.prototype.clone = function()
{
  var newObj = (this instanceof Array) ? [] : {};
  for (i in this)
  {
    if (i == 'clone') continue;
    if (this[i] && typeof this[i] == "object")
    {
      newObj[i] = this[i].clone();
    }
    else
      newObj[i] = this[i]
  }
  return newObj;
}

function identityMatrix()
{
  return [
          [1, 0, 0, 0],
          [0, 1, 0, 0],
          [0, 0, 1, 0],
          [0, 0, 0, 1]
        ];
}

/* Multiply two matricies */
function matrixMultiply(m1, m2)
{
  var result = identityMatrix();

  var cols = m1[0].length;
  var rows = m1.length;

  if (cols != m2.length)
    alert("matrix size error");

  for (var x = 0; x < cols; x++)
  {
    for (var y = 0; y < rows; y++)
    {
      var sum = 0;
      for (var z = 0; z < cols; z++)
      {
        sum += m1[y][z] * m2[z][x];
      }

      result[y][x] = sum;
    }
  }
  return result;
}

/* Generate a matrix to rotate a 3D vector. */
function genRotateMatrix(x, y, z)
{
  var result = identityMatrix();

  if (z)
  {
    var cosZ = Math.cos(-z);
    var sinZ = Math.sin(-z);
    var rotZ = [
                [cosZ, -sinZ, 0, 0],
                [sinZ,  cosZ, 0, 0],
                [   0,     0, 1, 0],
                [   0,     0, 0, 1]
               ];

    result = matrixMultiply(result, rotZ);
  }

  if (y)
  {
    var cosY = Math.cos(-y);
    var sinY = Math.sin(-y);
    var rotY = [
                [ cosY, 0, sinY, 0],
                [    0, 1,    0, 0],
                [-sinY, 0, cosY, 0],
                [    0, 0,    0, 1]
               ];

    result = matrixMultiply(result, rotY);
  }

  if (x)
  {
    var cosX = Math.cos(-x);
    var sinX = Math.sin(-x);
    var rotX = [
                [1,    0,     0, 0],
                [0, cosX, -sinX, 0],
                [0, sinX,  cosX, 0],
                [0,    0,     0, 1]
               ];
    result = matrixMultiply(result, rotX);
  }
  return result;
}

/* Generate a matrix to translate (move) a 3D vector  */
function genTranslateMatrix(x, y, z)
{
  return [
           [1, 0, 0, 0],
           [0, 1, 0, 0],
           [0, 0, 1, 0],
           [x, y, z, 1]
         ];
}

/* Transforms a point using the current transformation matrix */
function transformPoint(point, xfrm)
{
  var p = [[point.x, point.y, point.z, 1]];
  var pm = matrixMultiply(p, xfrm);

  point.x = pm[0][0];
  point.y = pm[0][1];
  point.z = pm[0][2];
}

/* =========================================================
 * Generate the Normal to a plane, given 3 points (3D)
 * which define a plane.
 * The vector returned starts at 0,0,0
 * is 10 units long in direction perpendicular to the plane.
 * Calculates A X B where p3-p1=A, p2-p1=B
 * ---------------------------------------------------------
 */
function _normal(p1, p2, p3)
{
  var a = new Point(p3.x-p1.x, p3.y-p1.y, p3.z-p1.z);   // vector from p1 to p3
  var b = new Point(p2.x-p1.x, p2.y-p1.y, p2.z-p1.z);   // vector from p1 to p2
  // a and b lie in the plane, a x b (cross product) is normal to both ie normal to plane
  // left handed coord system use left hand to get X product direction
  var nx = a.y*b.z - a.z*b.y;
  var ny = a.z*b.x - a.x*b.z;
  var nz = a.x*b.y - a.y*b.x;
  var mag = 0.1*Math.sqrt(nx*nx + ny*ny + nz*nz);  // make 'unit' vector 10px long

  var n = new Point(nx/mag, ny/mag, nz/mag);

  return n
}

/* =====================================================================
 * A 3d coordinate (left handed system)
 *
 * X +ve right
 * Y +ve up
 * Z +ve into screen
 * --------------------------------------------------------------------
 */
function Point(x, y, z)
{
  this.x = x;
  this.y = y;
  this.z = z;

  // Translated, rotated
  this.tx;
  this.ty;
  this.tz;

  // tx, ty, tz, projected to 2D as seen from viewpoint
  this.fx;
  this.fy;
}


/* =============================================================
 * DrawCmd3D
 * - drawFn: String, the canvas draw command name
 * - cPts: Array, [Point, Point ...] Bezier curve control points
 * - ep: Point, end point of the drawFn
 *--------------------------------------------------------------
 */
function DrawCmd3D(cmdStr, controlPoints, endPoint)
{
  this.drawFn = cmdStr;                      // String version of the canvas command to call
  this.cPts = controlPoints || [];           // array of parameters to pass to drawFn
  this.ep = endPoint;
}

/* ======================================================================
 * DrawCmd is a HTML5 canvas draw command in the from of a string
 * this.cmdStr (eg "moveTo") and a set of 2D coords 'this.parms' to pass
 * to the command (eg [5, 20] the pen will move to 5,20 in canvas coords.
 * ----------------------------------------------------------------------
 */
if (typeof DrawCmd == 'undefined')           // avoid clash with DrawCmd from 2D canvas graphics
{
  function DrawCmd(cmdStr, coords)
  {
    this.drawFn = cmdStr;                    // String version of the canvas command to call
    this.parms = coords || [];               // array of parameters to pass to drawFn
  }
}

/* ======================================================================
 * Path3D
 * Parameters:
 * - commands: Array of DrawCmd3D objects saved in 'drawCmds', a reference
 * to each DrawCmd3D vertex, and control point is stored in ppPts[].
 * - closed: boolean - true: path is filled, false: draw as wireframe
 * - color: String - optional HTML Color format for color
 *
 * Properties
 * - ppPts: is an array holding a references to all coords of drawCmds to
 * facilitate 3D transformations (rotate, translate, projection to 2D).
 * The transformation matrix 'xfrm' accumulates all the 3D trnalate and
 * rotate movement applied to the Path3D. Prior to drawing each point in
 * the ppPts array (and thus the drawCmds coords) is multiplied by this
 * transform matrix.
 * - drawCmds[]: is an array of DrawCmd3D objects desribing how to draw
 * the path, a DrawCmd3D object has components [String, [Points], Point]
 * where the String is a HTML5 canvas format draw function eg. 'lineTo',
 * 'bezierCurveTo' etc. the Points[] is a set of Bezier control points.
 * The last Point is the end coordinate of the path segment.
 * - color: String - optional HTML Color format for color
 * - closed: boolean - true: path is filled, false: draw as wireframe
 * - drawData: array holds the final 2D commands. Each element has
 * the same canvas draw command as drawCmds and the parameters are the 2D
 * projections of the drawCmds 3D points (after any transformations).
 * - origin:Point object - calculated as the average of x, y and z coordinates
 * of all vertices. Used as the reference point for sorting the Path3Ds of a
 * group prior to rendering
 * - normal: Point object - the vector from the origin to the normal point is
 * perpendicular to the plane of the Path3D
 * ----------------------------------------------------------------------
 */
function Path3D(commands, closed, col)
{
  this.ppPts = [];                 // references to all coords for transforming this path
  this.drawCmds = commands || [];  // array of DrawCmd3D objects
  this.color = null;               // use penColor if not specified
  this.closed = closed || false;   // draw as wirefame unless 'closed' = true
  this.drawData = [];              // 2D draw commands in format {string, [points]}
  this.origin = new Point(0, 0, 0);
  this.normal = new Point(0, 0, 0);

  var newCol = new RGBAColor(col);

  if (newCol.ok)
    this.color = newCol.toRGBA();                                // save as a string

  for (var i=0; i<this.drawCmds.length; i++)                     // step through each segment
  {
    for (var j=0; j<this.drawCmds[i].cPts.length; j++)           // step thru control points
    {
      this.ppPts[this.ppPts.length] = this.drawCmds[i].cPts[j];  // save a reference
    }
    if (this.drawCmds[i].ep != undefined)                        // closePath has no end point
      this.ppPts[this.ppPts.length] = this.drawCmds[i].ep;       // save reference to end point
  }

  var xSum = 0;
  var ySum = 0;
  var numPts = 0;    // total point counter for all commands
  for (i=0; i<this.drawCmds.length; i++)
  {
    if (this.drawCmds[i].ep != undefined)  // check for Z command, has no coords
    {
      xSum += this.drawCmds[i].ep.x;
      ySum += this.drawCmds[i].ep.y;
      numPts++;
    }
  }
  this.origin.x = xSum/numPts;
  this.origin.y = ySum/numPts;
  this.origin.z = 0;

  if (this.drawCmds.length < 2)
    return;
  // make the normal = aXb, a is vector from origin to data[0], b origin to data[1]
  var a = this.drawCmds[0].ep;    // get the start point of path
  var b = this.drawCmds[1].ep;    // get the 2nd point of path
  this.normal = _normal(this.origin, b, a);
  // NOTE: traverse CW normal is out of screen (-z), traverse path CCW normal is into screen (+z)

  // move normal from 0,0 to start from this.origin so back culling can test which is in front
  this.normal.x += this.origin.x;
  this.normal.y += this.origin.y;
  this.normal.z += this.origin.z;
}

Path3D.prototype.dup = function()
{
  var newObj = new Path3D();
  var i, j;

  newObj.ppPts = [];
  newObj.drawCmds = this.drawCmds.clone();
  newObj.color = this.color;    // use penColor if not specified
  newObj.closed = this.closed;     // draw as wirefame unless 'closed' = true
  newObj.drawData = this.drawData.clone();
  newObj.origin = this.origin.clone();
  newObj.normal = this.normal.clone();

  // Now ppPts is a set of references to the control points and vertices
  // so don't clone, make a new set of references.
  for (i=0; i < newObj.drawCmds.length; i++)  // step through the 2D drawing segments
  {
    for (j=0; j < newObj.drawCmds[i].cPts.length; j++)  // step thru control points
    {
      newObj.ppPts[newObj.ppPts.length] = newObj.drawCmds[i].cPts[j]; // save a reference
    }
    if (newObj.drawCmds[i].ep != undefined)           // closePath has no end point
      newObj.ppPts[newObj.ppPts.length] = newObj.drawCmds[i].ep; // save reference to end point
  }

  return newObj;
}

/* ====================================================================
 * appendPath3D concatinates 2 Path3Ds.
 * Use this to build complex Path3Ds segment by segment.
 * Parameters:
 * extPP: - the Path3D to be appended
 * delMove: - true: delete the inital Move DrawCmd3D. Each Path3D
 *            commands array starts with a Move. But if pen is already
 *            at the point Move is redundent
 * -------------------------------------------------------------------- */
Path3D.prototype.appendPath3D = function(extPP, delMove)
{
  var xSum = 0;
  var ySum = 0;
  var zSum = 0;
  var start = 0;

  if (delMove != undefined)
  {
    if (delMove===true)
      start = 1;
  }

  // extPP had its own array of points, these must be concatinated with the basePP
  // concat the points NOTE concat doesn't work
  for (i=start; i<extPP.ppPts.length; i++)
    this.ppPts[this.ppPts.length] = extPP.ppPts[i];
  for (i=start; i<extPP.drawCmds.length; i++)
    this.drawCmds[this.drawCmds.length] = extPP.drawCmds[i];

  // recalculate the origin and normal
  var numPts = 0;    // total point counter for all commands
  for (i=0; i<this.drawCmds.length; i++)
  {
    if (this.drawCmds[i].ep != undefined)
    {
      xSum += this.drawCmds[i].ep.x;
      ySum += this.drawCmds[i].ep.y;
      zSum += this.drawCmds[i].ep.z;
      numPts++;
    }
  }
  this.origin.x = xSum/numPts;
  this.origin.y = ySum/numPts;
  this.origin.z = zSum/numPts;

  var p2 = this.drawCmds[0].ep;    // get the start point of path
  var p3 = this.drawCmds[1].ep;    // get the 2nd point of path
  this.normal = _normal(this.origin, p3, p2);   // over-write the old value

  // make the normal start from the origin so back culling can test which is in front
  this.normal.x += this.origin.x;
  this.normal.y += this.origin.y;
  this.normal.z += this.origin.z;
}

/* add a single DrawCmd3D object to an existing polyPath */
Path3D.prototype.addDrawCmd = function(drawcmd)
{
  var xSum = 0;
  var ySum = 0;
  var zSum = 0;

  // concat the points NOTE concat doesn't work
  for (var i=0; i<drawcmd.cPts.length; i++)
    this.ppPts[this.ppPts.length] = drawcmd.cPts[i];
  // copy the end point too
  this.ppPts[this.ppPts.length] = drawcmd.ep;
  // now append the drawcmd to the drawCmds array
  this.drawCmds[this.drawCmds.length] = drawcmd;

  if (this.drawCmds.length < 3)  // need 3 points to get a non degenerate origin
    return;
  // recalculate the origin and normal
  var numPts = 0;    // total point counter for all commands
  for (i=0; i<this.drawCmds.length; i++)
  {
    // last parm of any drawCmd is the end point (others may be control points not on the plane)
    if (this.drawCmds[i].ep != undefined)
    {
      xSum += this.drawCmds[i].ep.x;
      ySum += this.drawCmds[i].ep.y;
      zSum += this.drawCmds[i].ep.z;
      numPts++;
    }
  }
  this.origin.x = xSum/numPts;
  this.origin.y = ySum/numPts;
  this.origin.z = zSum/numPts;

  var p2 = this.drawCmds[0].ep;    // get the start point of path
  var p3 = this.drawCmds[1].ep;    // get the 2nd point of path
  this.normal = _normal(this.origin, p3, p2);   // over-write the old value

  // make the normal start from the origin so back culling can test which is in front
  this.normal.x += this.origin.x;
  this.normal.y += this.origin.y;
  this.normal.z += this.origin.z;
}

Path3D.prototype.setColor = function(color)
{
  var col = new RGBAColor(color);

  if (col.ok)
    this.color = col.toRGBA();
}

/*=========================================================
 * rotate
 * Generate a transform matrix to rotate a point in 3D
 * Then multiply every point in ppPts and the origin and
 * normal by this matrix.
 *---------------------------------------------------------
 */
Path3D.prototype.rotate = function(xRot, yRot, zRot)
{
  var t = Math.PI/180.0;

  var xfrm = genRotateMatrix(xRot*t, yRot*t, zRot*t);

  for (var i=0; i<this.ppPts.length; i++)
  {
    transformPoint(this.ppPts[i], xfrm);
  }
  transformPoint(this.origin, xfrm);
  transformPoint(this.normal, xfrm);
}

/*=========================================================
 * translate
 * Generate a transform matrix to translate a point in 3D
 * Then multiply every point in ppPts and the origin and
 * normal by this matrix.
 *---------------------------------------------------------
 */
Path3D.prototype.translate = function(x, y, z)
{
  var xfrm = genTranslateMatrix(x, y, z);

  for (var i=0; i<this.ppPts.length; i++)
  {
    transformPoint(this.ppPts[i], xfrm);
  }
  transformPoint(this.origin, xfrm);
  transformPoint(this.normal, xfrm);
}

/*=========================================================
 * shadePath3D
 * Calculate a shade of this Path3Ds color given the
 * direction to the light source and the direction of the
 * viewpoint.
 * Parameters:
 * viewpointDistance: viewpoint is always on -z axis.
 *    hence viewpoint vector is 0,0,-viewpointDistance
 * sun: Point object of light souce position in 3D
 * defaultColour: RGBA color string, used if Path3D doesn't
 *    have color set.
 *---------------------------------------------------------
 */
Path3D.prototype.shadePath3D = function(viewpointDistance, sun, defaultColor)
{
  // viewpoint is always on -z axis, viewpointDistance from coord system origin
  var los = new Point(0, 0, -viewpointDistance);  // store this in the Line Of Sight vector
  var col = new RGBAColor(this.color);
  if (!col.ok)
    col = new RGBAColor(defaultColor);

  // luminence is dot product of shape normal and Sun vector
  var norm = 10*Math.sqrt(sun.x*sun.x + sun.y*sun.y + sun.z*sun.z);  // unit vectors are 10 long
  var lum = sun.x*(this.normal.tx-this.origin.tx) + sun.y*(this.normal.ty-this.origin.ty) + sun.z*(this.normal.tz-this.origin.tz);
  lum /= 2*norm;     // normalise to range 0..0.5
  lum = Math.abs(lum);     // normal can be up or down (back given same shading)
  lum += 0.3;          // add a base level so its not too dark

  /* Now calculate unit vector along line of sight to decide if we are looking at front or back
     if normal dot product with LOS is +ve its the top, -ve its the bottom
     bottom might get a different colour.
     no need to normalise, just need the sign of dot product */
  los.tx = los.x - this.origin.tx;
  los.ty = los.y - this.origin.ty;
  los.tz = los.z - this.origin.tz;
  if ((this.normal.tx-this.origin.tx)*los.tx + (this.normal.ty-this.origin.ty)*los.ty + (this.normal.tz-this.origin.tz)*los.tz < 0)
  {
    /*  looking at back,  make it magenta
    col.r = 200;
    col.g = 0;
    col.b = 200; */
    // back will be dark if normal (front) is pointing toward the sun
    if ((this.normal.tx-this.origin.tx)*sun.x + (this.normal.ty-this.origin.ty)*sun.y + (this.normal.tz-this.origin.tz)*sun.z > 0)
      lum = 0.3;
  }
  else
  {
    // looking at the front
    // front will be dark if normal is pointing away from sun
    if ((this.normal.tx-this.origin.tx)*sun.x + (this.normal.ty-this.origin.ty)*sun.y + (this.normal.tz-this.origin.tz)*sun.z < 0)
      lum = 0.3;
  }
   // calc rgb color based on V5 (component of normal to polygon in direction on POV)
  var cr = Math.round(lum*col.r);
  var cg = Math.round(lum*col.g);
  var cb = Math.round(lum*col.b);
  var ca = col.a;

  return "rgba("+cr+","+cg+","+cb+","+ca+")";     // string format 'rgba(r,g,b,a)'
}

/* 3D projection with perspective  */
Path3D.prototype.project3D = function(vpDistance, point)
{
  vpZ = -vpDistance;                      //
  // perspective projection
  point.fx = point.x * vpZ/(vpZ - point.z);
  point.fy = point.y * vpZ/(vpZ - point.z);
}


/* ========================================================================
 * Group3D is an object holding a set of Path3D objects.
 * If the Path3D is only to be drawn if its facing the viewpoint, as is the
 * case for solid shapes, the Path3Ds are stroed in 'backcullPaths'
 * If the Path3Ds are to be drawn regardless of orientation they are stored
 * in 'paths'.
 * The translate and rotate requests are applied to the xfrm matrix and this
 * is applied to the paths and backcullPaths (origins and normals) prior
 * to rendering.
 * All Path3Ds must be transformed from 3D to 2D by perspective projection
 * prior to drawing and the Path3Ds in backcullPaths must be culled when
 * this is done, the resulting 2D canvas draw commands are stored in
 * pathDrawList array and these are ready to be drawn onto a canvas.
 * ------------------------------------------------------------------------
 */
function Group3D()
{
  this.paths = [];                   // Path3Ds in this grouping
  this.backcullPaths = [];           // Path3Ds forming solid 3D object (cull backfaces)
  this.xfrm = identityMatrix();      // Group3D transform matrix
  this.pathDrawList = [];
}

Group3D.prototype.addPath3D = function(newPoly)
{
  this.paths[this.paths.length] = newPoly;
}

Group3D.prototype.addBackcullPath3D = function(bcPath)    // bcPath is a Path3D
{
  this.backcullPaths[this.backcullPaths.length] = bcPath;
}

/* Reset the transform matrix to the identity matrix */
Group3D.prototype.resetTransform = function()
{
  this.xfrm = identityMatrix();
}

/* Transforms a point using the current transformation matrix
 * this is a soft transform the original numbers are not overwritten
 */
Group3D.prototype.transform = function(point)
{
  var p = [[point.x, point.y, point.z, 1]];
  var pm = matrixMultiply(p, this.xfrm);

  point.tx = pm[0][0];
  point.ty = pm[0][1];
  point.tz = pm[0][2];
}

/* =============================================================
 * rotation, translation and transform
 * Apply rotation or translation factors to the current xfrm
 * matrix. The xfrm matrix will accumulate all transforms. It is
 * not automatically reset. It can be reset to the identity
 * matrix using Group3D.resetTransform method.
 * The transform is non-destructive, the original points x, y, z
 * values are preserved, the transformed points are stored as
 * tx, ty, tz and these are 2D projected and rendered.
 *--------------------------------------------------------------
 */
Group3D.prototype.rotation = function(xRot, yRot, zRot)
{
  var t = Math.PI/180.0;

  var rot = genRotateMatrix(xRot*t, yRot*t, zRot*t);
  this.xfrm = matrixMultiply(this.xfrm, rot);
}

/* Apply a translation to current transformation matrix  */
Group3D.prototype.translation = function(x, y, z)
{
  var trns = genTranslateMatrix(x, y, z);
  this.xfrm = matrixMultiply(this.xfrm, trns);
}

/*=========================================================
 * rotate
 * Generate transformation matrix to rotate a 3D point.
 * Then multiply every point in each Path3D its origin
 * and normal by this matrix.
 * The transformed x,y,z values overwrite the current
 * values. These relative movements should be used in shape
 * construction not animation.
 * NOTE: 'rotate' is different from 'rotation', which
 * does not overwrite the x,y,z coords of each point.
 *---------------------------------------------------------
 */
Group3D.prototype.rotate = function(xRot, yRot, zRot)
{
  var t = Math.PI/180.0;
  var i, j, poly;

  var xfrm = genRotateMatrix(xRot*t, yRot*t, zRot*t);

  // must transform all the paths normals and origins
  for (i=0; i<this.paths.length; i++)
  {
    poly = this.paths[i];
    for (j=0; j<poly.ppPts.length; j++)
    {
      transformPoint(poly.ppPts[j], xfrm);
    }
    transformPoint(poly.origin, xfrm);    // translate, rotate the origin of the ployPath
    transformPoint(poly.normal, xfrm);    // rotate the normal of each polygon
  }
  // must transform all the backcullPaths normals and origins
  for (i=0; i<this.backcullPaths.length; i++)
  {
    poly = this.backcullPaths[i];
    for (j=0; j<poly.ppPts.length; j++)
    {
      transformPoint(poly.ppPts[j], xfrm);
    }
    transformPoint(poly.origin, xfrm);    // translate, rotate the origin of the ployPath
    transformPoint(poly.normal, xfrm);    // rotate the normal of each polygon
  }
}

/*=========================================================
 * translate
 * Generate a transformation matrix to translate a 3D point
 * Then multiply every point in each Path3D, its origin
 * and normal by this matrix.
 * The transformed x,y,z values overwrite the current
 * values. These relative movements should be used in shape
 * construction not animation.
 * NOTE: 'rotate' is different from 'rotation', which
 * does not overwrite the x,y,z coords of each point.
 *---------------------------------------------------------
 */
Group3D.prototype.translate = function(x, y, z)
{
  var i, j, poly;

  var xfrm = genTranslateMatrix(x, y, z);

  // must transform all the paths normals and origins
  for (i=0; i<this.paths.length; i++)
  {
    poly = this.paths[i];
    for (j=0; j<poly.ppPts.length; j++)
    {
      transformPoint(poly.ppPts[j], xfrm);
    }
    transformPoint(poly.origin, xfrm);    // translate, rotate the origin of the ployPath
    transformPoint(poly.normal, xfrm);    // rotate the normal of each polygon
  }
  // must transform all the backcullPaths normals and origins
  for (i=0; i<this.backcullPaths.length; i++)
  {
    poly = this.backcullPaths[i];
    for (j=0; j<poly.ppPts.length; j++)
    {
      transformPoint(poly.ppPts[j], xfrm);
    }
    transformPoint(poly.origin, xfrm);    // translate, rotate the origin of the ployPath
    transformPoint(poly.normal, xfrm);    // rotate the normal of each polygon
  }
}

/* 3D projection with perspective  */
Group3D.prototype.project3D = function(viewpointDistance, point)
{
  // projection is onto screen at z = 0 viewpointDistance (NOT A coordinate)
  var r = -Math.abs(viewpointDistance);

  // perspective projection
  point.fx = point.tx * r/(r - point.tz);
  point.fy = point.ty * r/(r - point.tz);
}

/*==========================================================
 * genDrawData
 * Project the Group3D data points with perspective on 2D
 * surface (X,Y plane) cull any of the backcullPaths not
 * facing the viewpoint. Then sort the list of Paths
 * from back to front (position of the origin z values)
 * Returns the list of canvas draw commands
 *----------------------------------------------------------
 */
Group3D.prototype.genDrawData = function(viewpointDistance, wireframe)
{
  var h, i, j, k;
  var poly;
  // viewpoint is always on -z axis, viewpointDistance from coord system origin
  // store this in the line of sight vector
  var los = new Point(0, 0, -viewpointDistance);   // x,y,z hold observer location,

  // clear the paths drawlist
  this.pathDrawList = [];

  for (i=0; i<this.paths.length; i++)
  {
    poly = this.paths[i]; // convenience

    // do all the 3D transforms on the Group3D vertices
    for (j=0; j<poly.ppPts.length; j++)
    {
      this.transform(poly.ppPts[j]);       // rotate, translate Group3D in space
      this.project3D(viewpointDistance, poly.ppPts[j]);  // apply perspective to Group3D
    }
    this.transform(poly.origin);    // translate, rotate the origin of the Path3D
    this.project3D(viewpointDistance, poly.origin);  // apply perspective to origin
    this.transform(poly.normal);    // rotate the normal of each polygon
    this.project3D(viewpointDistance, poly.normal);  // apply perspective to normal

    // make the 2D drawCmd parameters array for each Path3D
    for(j=0; j<poly.drawCmds.length; j++)   // step through the draw segments
    {
      poly.drawData[j] = new DrawCmd(poly.drawCmds[j].drawFn);   // this is a 2D drawCmd
      for (k=0; k<poly.drawCmds[j].cPts.length; k++)   // extract flattened 2D coords from 3D Points
      {
        poly.drawData[j].parms[2*k] = poly.drawCmds[j].cPts[k].fx;
        poly.drawData[j].parms[2*k+1] = poly.drawCmds[j].cPts[k].fy;
      }
      // add the end point (check it exists since 'closePath' has no end point)
      if (poly.drawCmds[j].ep != undefined)
      {
        poly.drawData[j].parms[2*k] = poly.drawCmds[j].ep.fx;
        poly.drawData[j].parms[2*k+1] = poly.drawCmds[j].ep.fy;
      }
    }
    // add this Path3D to the draw list
    this.pathDrawList.push(poly);
  }

  for (i=0; i<this.backcullPaths.length; i++)
  {
    poly = this.backcullPaths[i];

    // do all the 3D transforms on the Group3D vertices
    for (j=0; j<poly.ppPts.length; j++)
    {
      this.transform(poly.ppPts[j]);       // rotate, translate Group3D in space
      this.project3D(viewpointDistance, poly.ppPts[j]);  // apply perspective to Group3D
    }
    this.transform(poly.origin);    // translate, rotate the origin of the Path3D
    this.project3D(viewpointDistance, poly.origin);  // apply perspective to origin
    this.transform(poly.normal);    // rotate the normal of each polygon
    this.project3D(viewpointDistance, poly.normal);  // apply perspective to normal

    // make the draw cmd parameters array for each Path3D
    for(j=0; j<poly.drawCmds.length; j++)   // step through the draw segments
    {
      poly.drawData[j] = new DrawCmd(poly.drawCmds[j].drawFn);   // this is a 2D drawCmd
      for (k=0; k<poly.drawCmds[j].cPts.length; k++)   // extract flattened 2D coords from 3D Points
      {
        poly.drawData[j].parms[2*k] = poly.drawCmds[j].cPts[k].fx;
        poly.drawData[j].parms[2*k+1] = poly.drawCmds[j].cPts[k].fy;
      }
      // add the end point (check it exists since 'closePath' has no end point)
      if (poly.drawCmds[j].ep != undefined)
      {
        poly.drawData[j].parms[2*k] = poly.drawCmds[j].ep.fx;
        poly.drawData[j].parms[2*k+1] = poly.drawCmds[j].ep.fy;
      }
    }

    // now do some backface culling while populating pathDrawList
    if (wireframe)      // backface culling for solid Group3Ds
    {
      this.pathDrawList.push(poly);
    }
    else
    {
      // calc line of sight vector (from viewpoint to Path3D origin)
      // no need to normalise, just need the sign of dot product
      los.tx = los.x - poly.origin.tx;
      los.ty = los.y - poly.origin.ty;
      los.tz = los.z - poly.origin.tz;
      if ((poly.normal.tx-poly.origin.tx)*los.tx + (poly.normal.ty-poly.origin.ty)*los.ty + (poly.normal.tz-poly.origin.tz)*los.tz > 0)
        this.pathDrawList.push(poly);
 //   else:  don't draw its at the back
    }
  }

  // Depth sorting (painters algorithm, draw from the back to front)
  this.pathDrawList.sort(function (poly1, poly2)
                          {
                            return poly2.origin.tz - poly1.origin.tz;
                          });
}
